<?php
/**
 * Activation handler
 *
 * @package     EDD\ActivationHandler
 * @since       1.0.0
 */

// Exit if accessed directly!
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * EDD Extension Activation Handler Class
 *
 * @since       1.0.0
 */
class EDD_Extension_Activation {

	/**
	 * Name of current plugin.
	 *
	 * @var         EDD_Extension_Activation $plugin_name
	 * @since       1.0.0
	 */
	public $plugin_name;

	/**
	 *  The path to main plugin file
	 *
	 * @var         EDD_Extension_Activation $plugin_path
	 * @since       1.0.0
	 */
	public $plugin_path;

	/**
	 * The main plugin file
	 *
	 * @var         EDD_Extension_Activation $plugin_file
	 * @since       1.0.0
	 */
	public $plugin_file;

	/**
	 * Whether or not EDD is present
	 *
	 * @var         EDD_Extension_Activation $has_edd
	 * @since       1.0.0
	 */
	public $has_edd;

	/**
	 * Path to EDD Base
	 *
	 * @var         EDD_Extension_Activation $edd_base
	 * @since       1.0.0
	 */
	public $edd_base;

	/**
	 * Setup the activation class
	 *
	 * @param       string $plugin_path Path of current plugin.
	 * @param       string $plugin_file Current plugin main file.
	 * @access      public
	 * @since       1.0.0
	 * @return      void
	 */
	public function __construct( $plugin_path, $plugin_file ) {
		// We need plugin.php!
		require_once ABSPATH . 'wp-admin/includes/plugin.php';

		$plugins = get_plugins();

		// Set plugin directory.
		$plugin_path       = array_filter( explode( '/', $plugin_path ) );
		$this->plugin_path = end( $plugin_path );

		// Set plugin file.
		$this->plugin_file = $plugin_file;

		// Set plugin name.
		if ( isset( $plugins[ $this->plugin_path . '/' . $this->plugin_file ]['Name'] ) ) {
			$this->plugin_name = str_replace( 'Easy Digital Downloads - ', '', $plugins[ $this->plugin_path . '/' . $this->plugin_file ]['Name'] );
		} else {
			$this->plugin_name = EDD_ENHANCED_SALES_REPORTS_NAME;
		}

		// Is EDD installed?
		foreach ( $plugins as $plugin_path => $plugin ) {
			if ( 'Easy Digital Downloads' === $plugin['Name'] ) {
				$this->has_edd  = true;
				$this->edd_base = $plugin_path;
				break;
			}
		}
	}


	/**
	 * Process plugin deactivation
	 *
	 * @access      public
	 * @since       1.0.0
	 * @return      void
	 */
	public function run() {
		// Display notice.
		add_action( 'admin_notices', array( $this, 'edd_enhanced_sales_reports_missing_edd_notice' ) );
	}


	/**
	 * Display notice if EDD isn't installed
	 *
	 * @access      public
	 * @since       1.0.0
	 * @return      void
	 */
	public function edd_enhanced_sales_reports_missing_edd_notice() {
		if ( $this->has_edd ) {
			$url  = esc_url( wp_nonce_url( admin_url( 'plugins.php?action=activate&plugin=' . $this->edd_base ), 'activate-plugin_' . $this->edd_base ) );
			$link = '<a href="' . $url . '">' . esc_html__( 'activate it', 'edd-extension-activation' ) . '</a>';
		} else {
			$url  = esc_url( wp_nonce_url( self_admin_url( 'update.php?action=install-plugin&plugin=easy-digital-downloads' ), 'install-plugin_easy-digital-downloads' ) );
			$link = '<a href="' . $url . '">' . esc_html__( 'install it', 'edd-extension-activation' ) . '</a>';
		}

		/* translators: %s is the link */
		echo '<div class="error"><p>' . esc_html( $this->plugin_name ) . sprintf( esc_html__( ' requires Easy Digital Downloads! Please %s to continue!', 'edd-extension-activation' ), esc_attr( $link ) ) . '</p></div>';
	}
}
